Option Explicit
Const VPinMAMEDriverVer = 3.10
'=======================
' VPinMAME driver core.
'=======================
' New in 3.10
'   - Public release
' New in 3.09
'   - Added new pulse switch handling for mechs
'   - Fixed bug in mech speed (thanks El Condor)
'   - The UserValue copying in vpmMoveBall sometimes causes a page fault
'     so I commented it out
' New in 3.08
'   - Added InitAddSnd to ballstack (sound to play when ball is added)
' New in 3.07
'   - vpmMoveBall now also copies UserValue
' New in 3.06
'   - Lights array can now contain arrays of lights
' New in 3.05
'   - Timer class can now call functions directly (via GetRef)
'   - Increased Lights array to 200 (Capcom got 2x88!)
'   - VLock now accepts small upwards move to allow almost horizontal layouts (A13)
'   - Removed .Pause in Light/Sol updates
' New in 3.04
'   - Noted that games use the last switch in a VLock control the gate.
'     All switches are now cleared when the gate opens instead of when the
'     ball leaves the kicker.
' New in 3.03
'   - Rewrite of the timer class. Fixed bug with fast pulsesw (spinners)
'     and it can now handle pulsing of any switch (e.g. switch 0)
'
' Put this at the top of the table file
'LoadVPM "02000000", "xxx.VBS", 3.10
'Const cGameName    = "xxxx" ' PinMAME short game name
'Const UseSolenoids = True
'Const UseLamps     = True
''Standard sound
'Const SSolenoidOn  = "SolOn"       'Solenoid activates
'Const SSolenoidOff = "SolOff"      'Solenoid deactivates
'Const SFlipperOn   = "FlipperUp"   'Flipper activated
'Const SFlipperOff  = "FlipperDown" 'Flipper deactivated
'Const SCoin        = "Quarter"     'Coin inserted
''Callbacks
'Set LampCallback   = GetRef("UpdateMultipleLamps")
'Set GICallback     = GetRef("UpdateGI")
'Set MotorCallback  = GetRef("UpdateMotors")
'
'Sub LoadVPM(VPMver, VBSfile, VBSver)
'	On Error Resume Next
'		If ScriptEngineMajorVersion < 5 Then MsgBox "VB Script Engine 5.0 or higher required"
'		ExecuteGlobal GetTextFile(VBSfile)
'		If Err Then MsgBox "Unable to open " & VBSfile & ". Ensure that it is in the same folder as this table. " & vbNewLine & Err.Description : Err.Clear
'		Set Controller = CreateObject("VPinMAME.Controller")
'		If Err Then MsgBox "Can't Load VPinMAME." & vbNewLine & Err.Description
'		If VPMver>"" Then If Controller.Version < VPMver Or Err Then MsgBox "VPinMAME ver " & VPMver & " required." : Err.Clear
'		If VPinMAMEDriverVer < VBSver Or Err Then MsgBox VBSFile & " ver " & VBSver & " or higher required."
'End Sub
'
'Sub Table_KeyDown(ByVal keycode)
'	If vpmKeyDown(keycode) Then Exit Sub
'	If keycode = PlungerKey Then Plunger.Pullback
'End Sub
'Sub Table_KeyUp(ByVal keycode)
'	If vpmKeyUp(keycode) Then Exit Sub
'	If keycode = PlungerKey Then Plunger.Fire
'End Sub
'
'Const cCredits  = ""
'Sub Table_Init
'	vpmInit Me
'	On Error Resume Next
'		With Controller
'			.GameName = cGameName
'			If Err Then MsgBox "Can't start Game" & cGameName & vbNewLine & Err.Description : Exit Sub
'			.SplashInfoLine = cCredits
'			.HandleMechanics = 0
'			.ShowDMDOnly = True : .ShowFrame = False : .ShowTitle = False
'			.Run : If Err Then MsgBox Err.Description
'		End With
'	On Error Goto 0
'' Nudging
'	vpmNudge.TiltSwitch = swTilt
'	vpmNudge.Sensitivity = 5
'	vpmNudge.TiltObj = Array(Bumper1,Bumper2,LeftslingShot,RightslingShot)
'' Map switches and lamps
'	vpmCreateEvents colSwObjects ' collection of triggers etc
'	vpmMapLights    colLamps     ' collection of all lamps
'' Trough handler
'	Set bsTrough = New cvpmBallStack
'	bsTrough.InitNoTrough BallRelease, swOuthole, 90, 2
'	'or
'	bsTrough.InitSw swOuthole,swTrough1,swTrough2,0,0,0,0
'---------------------------------------------------------------
Dim Controller   ' VPinMAME Controller Object
Dim vpmTimer     ' Timer Object
Dim vpmNudge     ' Nudge handler Object
Dim Lights(200)  ' Put all lamps in an array for easier handling
' If more than one lamp is connected fill this with an array of each light
Dim vpmMultiLights() : ReDim vpmMultiLights(0)
Private gNextMechNo : gNextMechNo = 0 ' keep track of created mech handlers (would be nice with static members)

' Callbacks
Dim SolCallback(64) ' Solenoids (parsed at Runtime)
Dim LampCallback    ' Called after lamps are updated
Dim GICallback      ' Called for each changed GI String
Dim MotorCallback   ' Called after solenoids are updated
Dim vpmCreateBall   ' Called whenever a vpm class needs to create a ball

' Game specific info
Dim ExtraKeyHelp    ' Help string for game specific keys
Dim vpmShowDips     ' Show DIPs function
'-----------------------------------------------------------------------------
' These helper function requires the following objects on the table:
'   PinMAMETimer   : Timer object
'   PulseTimer     : Timer object
'
' Available classes:
' ------------------
' cvpmTimer (Object = vpmTimer)
'   (Public)  .PulseSwitch   - pulse switch and call callback after delay (default)
'   (Public)  .PulseSw       - pulse switch
'   (Public)  .AddTimer      - call callback after delay
'   (Public)  .Reset         - Re-set all ballStacks
'   (Friend)  .InitTimer     - initialise fast or slow timer
'   (Friend)  .EnableUpdate  - Add/remove automatic update for an instance
'   (Private) .Update        - called from slow timer
'   (Private) .FastUpdate    - called from fast timer
'   (Friend)  .AddResetObj   - Add object that needs to catch reset
'
' cvpmBallStack (Create as many as needed)
'   (Public) .InitSw        - init switches used in stack
'   (Public) .InitSaucer    - init saucer
'   (Public) .InitNoTrough  - init a single ball, no trough handler
'   (Public) .InitKick      - init exit kicker
'   (Public) .InitAltKick   - init second kickout direction
'   (Public) .CreateEvents  - Create addball events for kickers
'   (Public) .KickZ         - Z axis kickout angle (radians)
'   (Public) .KickBalls     - Maximum number of balls kicked out at the same time
'   (Public) .KickForceVar  - Initial ExitKicker Force value varies by this much (+/-, minimum force = 1)
'   (Public) .KickAngleVar  - ExitKicker Angle value varies by this much (+/-)
'   (Public) .BallColour    - Set ball colour
'   (Public) .TempBallImage  - Set ball image for next ball only
'   (Public) .TempBallColour - Set ball colour for next ball only
'   (Public) .BallImage     - Set ball image
'   (Public) .InitAddSnd    - Sounds when ball enters stack
'   (Public) .InitEntrySnd  - Sounds for Entry kicker
'   (Public) .InitExitSnd   - Sounds for Exit kicker
'   (Public) .AddBall       - add ball in "kicker" to stack
'   (Public) .SolIn         - Solenoid handler for entry solenoid
'   (Public) .EntrySol_On   - entry solenoid fired
'   (Public) .SolOut        - Solenoid handler for exit solenoid
'   (Public) .SolOutAlt     - Solenoid handler for exit solenoid 2nd direction
'   (Public) .ExitSol_On    - exit solenoid fired
'   (Public) .ExitAltSol_On - 2nd exit solenoid fired
'   (Public) .Balls         - get/set number of balls in stack (default)
'   (Public) .BallsPending  - get number of balls waiting to come in to stack
'   (Public) .IsTrough      - Specify that this is the main ball trough
'   (Public) .Reset         - reset and update all ballstack switches
'   (Friend) .Update        - Update ball positions (from vpmTimer class)
'  Obsolete
'   (Public) .SolExit       - exit solenoid handler
'   (Public) .SolEntry      - Entry solenoid handler
'   (Public) .InitProxy     - Init proxy switch

' cvpmNudge (Object = vpmNudge)
'   Hopefully we can add a real pendulum simulator in the future
'   (Public)  .TiltSwitch   - set tilt switch
'   (Public)  .Senitivity   - Set tiltsensitivity (0-10)
'   (Public)  .TiltObj      - Set objects affected by tilt
'   (Public)  .DoNudge dir,power  - Nudge table
'   (Public)  .SolGameOn    - Game On solenoid handler
'   (Private) .Update       - Handle tilting
'
' cvpmDropTarget (create as many as needed)
'   (Public)  .InitDrop     - initialise DropTarget bank
'   (Public)  .CreateEvents - Create Hit events
'   (Public)  .InitSnd      - sound to use for targets
'   (Public)  .AnyUpSw      - Set AnyUp switch
'   (Public)  .AllDownSw    - Set all down switch
'   (Public)  .AllDown      - All targets down?
'   (Public)  .Hit          - A target had been hit
'   (Public)  .SolHit       - Solenoid handler for dropping a target
'   (Public)  .SolUnHit     - Solenoid handler for raising a target
'   (Public)  .SolDropDown  - Solenoid handler for Bank down
'   (Public)  .SolDropUp    - Solenoid handler for Bank reset
'   (Public)  .DropSol_On   - Reset target bank
'   (Friend)  .SetAllDn     - check alldown & anyup switches
'
' cvpmMagnet (create as many as needed)
'   (Public)  .InitMagnet   - initialise magnet
'   (Public)  .CreateEvents - Create Hit/Unhit events
'   (Public)  .Solenoid     - Set solenoid that controls magnet
'   (Public)  .GrabCenter   - Magnet grabs ball at center
'   (Public)  .MagnetOn     - Turn magnet on and off
'   (Public)  .X            - Move magnet
'   (Public)  .Y            - Move magnet
'   (Public)  .Strength     - Change strength
'   (Public)  .Size         - Change magnet reach
'   (Public)  .AddBall      - A ball has come withing range
'   (Public)  .RemoveBall   - A ball is out of reach for the magnet
'   (Public)  .Balls        - Balls currently within magnets reach
'   (Public)  .AttractBall  - attract ball to magnet
'   (Private) .Update       - update all balls (called from timer)
'   (Private) .Reset        - handle emulation reset
'  Obsolete
'   (Public)  .Range        - Change magnet reach

' cvpmTurnTable (create as many as needed)
'   (Public)  .InitTurnTable - initialise turntable
'   (Public)  .CreateEvents  - Create Hit/Unhit events
'   (Public)  .MaxSpeed      - Maximum speed
'   (Public)  .SpinUp        - Speedup acceleration
'   (Public)  .SpinDown      - Retardation
'   (Public)  .Speed         - Current speed
'   (Public)  .MotorOn       - Motor On/Off
'   (Public)  .SpinCW        - Control direction
'   (Public)  .SolMotorState - Motor on/off solenoid handler
'   (Public)  .AddBall       - A ball has come withing range
'   (Public)  .RemoveBall    - A ball is out of reach for the magnet
'   (Public)  .Balls         - Balls currently within magnets reach
'   (Public)  .AffectBall    - affect a ball
'   (Private) .Update        - update all balls (called from timer)
'   (Private) .Reset         - handle emulation reset

' cvpmMech (create as many as needed)
'   (Public)  .Sol1, Sol2    - Controlling solenoids
'   (Public)  .MType         - type of mechanics
'   (Public)  .Length, Steps
'   (Public)  .Acc, Ret      - Acceleration, retardation
'   (Public)  .AddSw         - Automatically controlled switches
'   (Public)  .AddPulseSw    - Automatically pulsed switches
'   (Public)  .Callback      - Update graphics function
'   (Public)  .Start         - Start mechanics handler
'   (Public)  .Position      - Current position
'   (Public)  .Speed         - Current Speed
'   (Private) .Update
'   (Private) .Reset
'
' cvpmCaptiveBall (create as many as needed)
'   (Public)  .InitCaptive   - Initialise captive balls
'   (Public)  .CreateEvents  - Create events for captive ball
'   (Public)  .ForceTrans    - Amount of force tranferred to captive ball (0-1)
'   (Public)  .MinForce      - Minimum force applied to the ball
'   (Public)  .NailedBalls   - Number of "nailed" balls infront of captive ball
'   (Public)  .RestSwitch    - Switch activated when ball is in rest position
'   (Public)  .Start         - Create moving ball etc.
'   (Public)  .TrigHit       - trigger in front of ball hit (or unhit)
'   (Public)  .BallHit       - Wall in front of ball hit
'   (Public)  .BallReturn    - Captive ball has returned to kicker
'   (Private) .Reset
'
' cvpmVLock (create as many as needed)
'   (Public)  .InitVLock     - Initialise the visible ball stack
'   (Public)  .ExitDir       - Balls exit angle (like kickers)
'   (Public)  .ExitForce     - Force of balls kicked out
'   (Public)  .KickForceVar  - Vary kickout force
'   (Public)  .InitSnd       - Sounds to make on kickout
'   (Public)  .Balls         - Number of balls in Lock
'   (Public)  .SolExit       - Solenoid event
'   (Public)  .CreateEvents  - Create events needed
'   (Public)  .TrigHit       - called from trigger hit event
'   (Public)  .TrigUnhit     - called from trigger unhit event
'   (Public)  .KickHit       - called from kicier hit event
'
' cvpmDips (create as many as needed)
'   (Public)  .AddForm       - create a form (AKA dialogue)
'   (Public)  .AddChk        - add a chckbox
'   (Public)  .AddChkExtra   -   -  "" -     for non-dip settings
'   (Public)  .AddFrame      - add a frame with checkboxes or option buttons
'   (Public)  .AddFrameExtra -  - "" - for non-dip settings
'   (Public)  .AddLabel      - add a label (text string)
'   (Public)  .ViewDips      - Show form
'   (Public)  .ViewDipsExtra -  - "" -  with non-dip settings
'
' Generic solenoid handlers:
' --------------------------
' vpmSolFlipper flipObj1, flipObj2  - "flips flippers". Set unused to Nothing
' vpmSolDiverter divObj, sound      - open/close diverter (flipper) with/without sound
' vpmSolWall wallObj, sound         - Raise/Drop wall with/without sound
' vpmSolToggleWall wall1, wall2, sound - Toggle between two walls
' vpmSolToggleObj obj1,obj2,sound   - Toggle any objects
' vpmSolAutoPlunger plungerObj      - Autoplunger/kickback
' vpmSolGate obj, sound             - Open/close gate
' vpmSolSound sound                 - Play sound only
' vpmFlasher flashObj               - Flashes flasher
'
' Generating events:
' ------------------
' vpmCreateEvents
' cpmCreateLights
'
' Variables declared (to be filled in):
' ---------------------------------------
' SolCallback()  - handler for each solenoid
' Lights()       - Lamps
'
' Constants used (must be defined):
' ---------------------------------
' UseSolenoids   - Update solenoids
' MotorCallback  - Called once every update for mechanics or custom sol handler
' UseLamps       - Update lamps
' LampCallback   - Sub to call after lamps are updated
'                  (or every update if UseLamps is false)
' GICallback     - Sub to call to update GI strings
' SFlipperOn     - Flipper activate sound
' SFlipperOff    - Flipper deactivate sound
' SSolenoidOn    - Solenoid activate sound
' SSolenoidOff   - Solenoid deactivate sound
' SCoin          - Coin Sound
' ExtraKeyHelp   - Game specific keys in help window
'
' Exported variables:
' -------------------
' vpmTimer      - Timer class for PulsSwitch etc
' vpmNudge      - Class for table nudge handling
'-----------------------------------------------------
Private Const PinMameInterval = 1

Private Const conStackSw    = 7  ' Stack switches
Private Const conMaxBalls   = 10
Private Const conMaxTimers  = 20 ' Spinners can generate a lot of timers
Private Const conTimerPulse = 40 ' Timer runs at 25Hz
Private Const conFastTicks  = 4  ' Fast is 4 times per timer pulse

Private Const conMaxSwHit   = 5  ' Don't stack up more that 5 events for each switch

' Dictionary
' Somehow Microsoft managed to make the dictionary object unsafe for scripting!
' To avoid security warnings I create my own dictionary
' To make it easier the key must be an object and the item must not be an object
Class cvpmDictionary
	Private cKey(), cItem(), cNext

	Private Sub Class_Initialize : cNext = 0 : ReDim cKey(0), cItem(0) : End Sub

	Private Function FindKey(aKey)
		Dim ii : FindKey = -1
		If cNext > 0 Then
			For ii = 0 To cNext - 1
				If cKey(ii) Is aKey Then FindKey = ii : Exit Function
			Next
		End If
	End Function

	Public Property Get Count : Count = cNext : End Property

	Public Property Get Item(aKey)
		Dim no : no = FindKey(aKey)
		If no >= 0 Then Item = cItem(no) Else Add aKey, Empty : Item = Empty
	End Property

	Public Property Let Item(aKey, aData)
		Dim no : no = FindKey(aKey)
		If no >= 0 Then cItem(no) = aData Else Add aKey, aData
	End Property

	Public Property Set Key(aKey)
		Dim no : no = FindKey(aKey)
		If no >= 0 Then Set cKey(no) = aKey Else Exit Property ' Err.Raise xxx
	End Property

	Public Sub Add(aKey, aItem)
		Dim no : no = FindKey(aKey)
		If no >= 0 Then ' already exists. Just change the value
			cItem(no) = aItem
		Else
			ReDim Preserve cKey(cNext), cItem(cNext)
			Set cKey(cNext) = aKey : cItem(cNext) = aItem : cNext = cNext + 1
		End If
	End Sub

	Public Sub Remove(aKey)
		Dim ii, no
		no = FindKey(aKey) : If no < 0 Then Exit Sub 'Err.Raise xxx
		cNext = cNext - 1
		If no < cNext Then
			For ii = no To cNext - 1 : Set cKey(ii) = cKey(ii+1) : cItem(ii) = cItem(ii+1) : Next
		End If
		ReDim Preserve cKey(cNext), cItem(cNext)
	End Sub

	Public Sub RemoveAll : ReDim cKey(0), cItem(0) : cNext = 0 : End Sub

	Public Function Exists(aKey) : Exists = FindKey(aKey) >= 0 : End Function

	Public Function Items : If cNext > 0 Then Items = cItem Else Items = Array() : End If : End Function

	Public Function Keys  : If cNext > 0 Then Keys  = cKey  Else Keys  = Array() : End If : End Function
End Class

'--------------------
'       Timer
'--------------------
Class cvpmTimer
	Private cQue, cNow, cTimers
	Private cSlowUpdates, cFastUpdates, cResets, cFastTimer

	Private Sub Class_Initialize
		ReDim cQue(conMaxTimers) : cNow = 0 : cTimers = 0
		Set cSlowUpdates = New cvpmDictionary : Set cFastUpdates = New cvpmDictionary : Set cResets = New cvpmDictionary
	End Sub

	Public Sub InitTimer(aTimerObj, aFast)
		If aFast Then
			Set cFastTimer = aTimerObj
			aTimerObj.TimerInterval = conTimerPulse \ conFastTicks
			aTimerObj.TimerEnabled = False
			vpmBuildEvent aTimerObj, "Timer", "vpmTimer.FastUpdate"
		Else
			aTimerObj.Interval = conTimerPulse : aTimerObj.Enabled = True
			vpmBuildEvent aTimerObj, "Timer", "vpmTimer.Update"
		End If
	End Sub

	Sub EnableUpdate(aClass, aFast, aEnabled)
		On Error Resume Next
		If aFast Then
			If aEnabled Then cFastUpdates.Add aClass, 0 : Else cFastUpdates.Remove aClass
			cFastTimer.TimerEnabled = cFastUpdates.Count > 0
		Else
			If aEnabled Then cSlowUpdates.Add aClass, 0 : Else cSlowUpdates.Remove aClass
		End If
	End Sub

	Public Sub Reset
		Dim obj : For Each obj In cResets.Keys : obj.Reset : Next
	End Sub

	Public Sub FastUpdate
		Dim obj : For Each obj In cFastUpdates.Keys : obj.Update : Next
	End Sub

	Public Sub Update
		Dim ii, jj, sw, obj

		For Each obj In cSlowUpdates.Keys : obj.Update : Next
		If cTimers = 0 Then Exit Sub
		cNow = cNow + 1 : ii = 1

		Do While ii <= cTimers
			If cQue(ii)(0) <= cNow Then
				If cQue(ii)(1) = 0 Then
					If isObject(cQue(ii)(3)) Then
						Call cQue(ii)(3)(cQue(ii)(2))
					ElseIf varType(cQue(ii)(3)) = vbString Then
						If cQue(ii)(3) > "" Then Execute cQue(ii)(3) & " " & cQue(ii)(2) & " "
					End If
					cTimers = cTimers - 1
					For jj = ii To cTimers : cQue(jj) = cQue(jj+1) : Next : ii = ii - 1
				ElseIf cQue(ii)(1) = 1 Then
					Controller.Switch(cQue(ii)(2)) = False
					cQue(ii)(0) = cNow + cQue(ii)(4) : cQue(ii)(1) = 0
				Else '2
					Controller.Switch(cQue(ii)(2)) = True
					cQue(ii)(1) = 1
				End If
			End If
			ii = ii + 1
		Loop
	End Sub

	Public Sub AddResetObj(aObj)  : cResets.Add aObj, 0 : End Sub

	Public Sub PulseSw(aSwNo) : PulseSwitch aSwNo, 0, 0 : End Sub

	Public Default Sub PulseSwitch(aSwNo, aDelay, aCallback)
		Dim ii, count, last
		count = 0
		For ii = 1 To cTimers
			If cQue(ii)(1) > 0 And cQue(ii)(2) = aSwNo Then count = count + 1 : last = ii
		Next
		If count >= conMaxSwHit Or cTimers = conMaxTimers Then Exit Sub
		cTimers = cTimers + 1 : cQue(cTimers) = Array(cNow, 2, aSwNo, aCallback, aDelay\conTimerPulse)
		If count Then cQue(cTimers)(0) = cQue(last)(0) + cQue(last)(1)
	End Sub

	Public Sub AddTimer(aDelay, aCallback)
		If cTimers = conMaxTimers Then Exit Sub
		cTimers = cTimers + 1
		cQue(cTimers) = Array(cNow + aDelay \ conTimerPulse, 0, 0, aCallback)
	End Sub
End Class

'--------------------
'     BallStack
'--------------------
Class cvpmBallStack
	Private cSw(), cEntrySw, cBalls, cBallIn, cBallPos(), cSaucer
	Private cInitKicker, cExitKicker, cExitDir, cExitForce
	Private cExitDir2, cExitForce2
	Private cEntrySnd, cEntrySndBall, cExitSnd, cExitSndBall, cAddSnd
	Public KickZ, KickBalls, KickForceVar, KickAngleVar

	Private Sub Class_Initialize
		ReDim cSw(conStackSw), cBallPos(conMaxBalls)
		cBallIn = 0 : cBalls = 0 : cExitKicker = 0 : cInitKicker = 0
		KickBalls = 1 : cSaucer = False : cExitDir = 0 : cExitForce = 0
		cExitDir2 = 0 : cExitForce2 = 0 : KickZ = 0 : KickForceVar = 0 : KickAngleVar = 0
		cAddSnd = 0 : cEntrySnd = 0 : cEntrySndBall = 0 : cExitSnd = 0 : cExitSndBall = 0
		vpmTimer.AddResetObj Me
	End Sub

	Private Property Let NeedUpdate(aEnabled) : vpmTimer.EnableUpdate Me, False, aEnabled : End Property

	Private Function SetSw(aNo, aStatus)
		SetSw = False : If aNo <= conStackSw Then If cSw(aNo) Then Controller.Switch(cSw(aNo)) = aStatus : SetSw = True
	End Function

	Public Sub Reset
		Dim ii : If cBalls Then For ii = 1 to cBalls : SetSw cBallPos(ii), True : Next
	End Sub

	Public Sub Update
		Dim BallQue, ii
		NeedUpdate = False : BallQue = 1
		For ii = 1 To cBalls
			If cBallpos(ii) > BallQue Then
				SetSw cBallPos(ii), False
				Do
					cBallPos(ii) = cBallPos(ii) - 1
				Loop Until SetSw(cBallPos(ii), True) Or cBallPos(ii) = BallQue
				NeedUpdate = True
			End If
			BallQue = cBallPos(ii) + 1
		Next
	End Sub

	Public Sub AddBall(aKicker)
		If isObject(aKicker) Then
			If cSaucer Then
				If aKicker Is cExitKicker Then
					cExitKicker.Enabled = False : cInitKicker = 0
				Else
					aKicker.Enabled = False : Set cInitKicker = aKicker
				End If
			Else
				aKicker.DestroyBall
			End If
		ElseIf cSaucer Then
			cExitKicker.Enabled = False : cInitKicker = 0
		End If
		If cEntrySw Then
			Controller.Switch(cEntrySw) = True : cBallIn = cBallIn + 1
		Else
			cBalls = cBalls + 1 : cBallPos(cBalls) = conStackSw + 1 : NeedUpdate = True
		End If
		PlaySound cAddSnd
	End Sub

	' A bug in the script engine forces the "End If" at the end
	Public Sub SolIn(aEnabled)     : If aEnabled Then KickIn        : End If : End Sub
	Public Sub SolOut(aEnabled)    : If aEnabled Then KickOut False : End If : End Sub
	Public Sub SolOutAlt(aEnabled) : If aEnabled Then KickOut True  : End If : End Sub
	Public Sub EntrySol_On   : KickIn        : End Sub
	Public Sub ExitSol_On    : KickOut False : End Sub
	Public Sub ExitAltSol_On : KickOut True  : End Sub

	Private Sub KickIn
		If cBallIn Then PlaySound cEntrySndBall Else PlaySound cEntrySnd : Exit Sub
		cBalls = cBalls + 1 : cBallIn = cBallIn - 1 : cBallPos(cBalls) = conStackSw + 1 : NeedUpdate = True
		If cEntrySw And cBallIn = 0 Then Controller.Switch(cEntrySw) = False
	End Sub

	Private Sub KickOut(aAltSol)
		Dim ii,jj, kForce, kDir, kBaseDir
		If cBalls Then PlaySound cExitSndBall Else PlaySound cExitSnd : Exit Sub
		If aAltSol Then kForce = cExitForce2 : kBaseDir = cExitDir2 Else kForce = cExitForce : kBaseDir = cExitDir
		kForce = kForce + (Rnd - 0.5)*KickForceVar
		If cSaucer Then
			SetSw 1, False : cBalls = 0 : kDir = kBaseDir + (Rnd - 0.5)*KickAngleVar
			If isObject(cInitKicker) Then
				vpmCreateBall cExitKicker : cInitKicker.Destroyball : cInitKicker.Enabled = True
			Else
				cExitKicker.Enabled = True
			End If
			cExitKicker.Kick kDir, kForce, KickZ
		Else
			For ii = 1 To kickballs
				If cBalls = 0 Or cBallPos(1) <> ii Then Exit For ' No more balls
				For jj = 2 To cBalls ' Move balls in array
					cBallPos(jj-1) = cBallPos(jj)
				Next
				cBallPos(cBalls) = 0 : cBalls = cBalls - 1 : NeedUpdate = True
				SetSw ii, False
				If isObject(cExitKicker) Then
					If kForce < 1 Then kForce = 1
					kDir = kBaseDir + (Rnd - 0.5)*KickAngleVar
					vpmTimer.AddTimer (ii-1)*200, "vpmCreateBall(" & cExitKicker.Name & ").Kick " &_
					  CInt(kDir) & "," & Replace(kForce,",",".") & "," & Replace(KickZ,",",".") & " '"
				End If
				kForce = kForce * 0.8
			Next
		End If
	End Sub

	Public Sub InitSaucer(aKicker, aSw, aDir, aPower)
		InitKick aKicker, aDir, aPower : cSaucer = True
		If aSw Then cSw(1) = aSw Else cSw(1) = aKicker.TimerInterval
	End Sub

	Public Sub InitNoTrough(aKicker, aSw, aDir, aPower)
		InitKick aKicker, aDir, aPower : Balls = 1
		If aSw Then cSw(1) = aSw Else cSw(1) = aKicker.TimerInterval
		If Not IsObject(vpmTrough) Then Set vpmTrough = Me
	End Sub

	Public Sub InitSw(aEntry, aSw1, aSw2, aSw3, aSw4, aSw5, aSw6, aSw7)
		cEntrySw = aEntry : cSw(1) = aSw1 : cSw(2) = aSw2 : cSw(3) = aSw3 : cSw(4) = aSw4
		cSw(5) = aSw5 : cSw(6) = aSw6 : cSw(7) = aSw7
		If Not IsObject(vpmTrough) Then Set vpmTrough = Me
	End Sub

	Public Sub InitKick(aKicker, aDir, aForce)
		Set cExitKicker = aKicker : cExitDir = aDir : cExitForce = aForce
	End Sub

	Public Sub CreateEvents(aName, aKicker)
		Dim obj, tmp
		If Not vpmCheckEvent(aName, Me) Then Exit Sub
		vpmSetArray tmp, aKicker
		For Each obj In tmp
			If isObject(obj) Then
				vpmBuildEvent obj, "Hit", aName & ".AddBall Me"
			Else
				vpmBuildEvent cExitKicker, "Hit", aName & ".AddBall Me"
			End If
		Next
	End Sub

	Public Property Let IsTrough(aIsTrough)
		If aIsTrough Then
			Set vpmTrough = Me
		ElseIf IsObject(vpmTrough) Then
			If vpmTrough Is Me Then vpmTrough = 0
		End If
	End Property

	Public Property Get IsTrough : IsTrough = vpmTrough Is Me : End Property

	Public Sub InitAltKick(aDir, aForce)
		cExitDir2 = aDir : cExitForce2 = aForce
	End Sub

	Public Sub InitEntrySnd(aBall, aNoBall) : cEntrySndBall = aBall : cEntrySnd = aNoBall : End Sub
	Public Sub InitExitSnd(aBall, aNoBall)  : cExitSndBall = aBall  : cExitSnd = aNoBall  : End Sub
	Public Sub InitAddSnd(aSnd) : cAddSnd = aSnd : End Sub

	Public Property Let Balls(aBalls)
		Dim ii
		For ii = 1 To conStackSw
			If cSw(ii) Then Controller.Switch(cSw(ii)) = False
			cBallPos(ii) = 8
		Next
		If cSaucer And aBalls > 0 And cBalls = 0 Then vpmCreateBall cExitKicker
		cBalls = aBalls : NeedUpdate = True
	End Property

	Public Default Property Get Balls : Balls = cBalls         : End Property
	Public Property Get BallsPending  : BallsPending = cBallIn : End Property

	' Obsolete stuff
	Public Sub SolEntry(aSnd1, aSnd2, aEnabled)
		If aEnabled Then cEntrySndBall = aSnd1 : cEntrySnd = aSnd2 : KickIn
	End Sub
	Public Sub SolExit(aSnd1, aSnd2, aEnabled)
		If aEnabled Then cExitSndBall = aSnd1 : cExitSnd = aSnd2 : KickOut False
	End Sub
	Public Sub InitProxy(aProxyPos, aSwNo) : End Sub
	Public TempBallColour, TempBallImage, BallColour
	Public Property Let BallImage(aImage) : vpmBallImage = aImage : End Property
End Class

'--------------------
'       Nudge
'--------------------
class cvpmNudge
	Private cCount, cSensitivity, cNudgeTimer, cSlingBump, cForce
	Public TiltSwitch

	Private Sub Class_Initialize
		cCount = 0 : TiltSwitch = 0 : cSensitivity = 5 : vpmTimer.AddResetObj Me
	End sub

	Private Property Let NeedUpdate(aEnabled) : vpmTimer.EnableUpdate Me, False, aEnabled : End Property

	Public Property Let TiltObj(aSlingBump)
		Dim ii
		If vpmVPVer >= 61 Then vpmSetArray cSlingBump, aSlingBump : Exit Property
		ReDim cForce(vpmSetArray(cSlingBump, aSlingBump))
		For ii = 0 To UBound(cForce)
			If TypeName(cSlingBump(ii)) = "Bumper" Then cForce(ii) = cSlingBump(ii).Force Else cForce(ii) = cSlingBump(ii).SlingshotStrength
		Next
	End Property

	Public Property Let Sensitivity(aSens) : cSensitivity = (10-aSens)+1 : End property

	Public Sub DoNudge(ByVal aDir, ByVal aForce)
		aDir = aDir + (Rnd-0.5)*15*aForce : aForce = (0.6+Rnd*0.8)*aForce
		Nudge aDir, aForce
		If TiltSwitch = 0 Then Exit Sub ' If no switch why care
		cCount = cCount + aForce * 1.2
		If cCount > cSensitivity + 10 Then cCount = cSensitivity + 10
		If cCount >= cSensitivity Then vpmTimer.PulseSw TiltSwitch
		NeedUpdate = True
	End sub

	Public Sub Update
		If cCount > 0 Then
			cNudgeTimer = cNudgeTimer + 1
			If cNudgeTimer > 1000\conTimerPulse Then
				If cCount > cSensitivity+1 Then cCount = cCount - 1 : vpmTimer.PulseSw TiltSwitch
				cCount = cCount - 1 : cNudgeTimer = 0
			End If
		Else
			cCount = 0 : NeedUpdate = False
		End If
	End Sub

	Public Sub Reset : cCount = 0 : End Sub

	Public Sub SolGameOn(aEnabled)
		Dim obj, ii
		If vpmVPVer >= 61 Then
			For Each obj In cSlingBump : obj.Disabled = Not aEnabled : Next
		ElseIf aEnabled Then
			ii = 0
			For Each obj In cSlingBump
				If TypeName(obj) = "Bumper" Then obj.Force = cForce(ii) Else obj.SlingshotStrength = cForce(ii)
				ii = ii + 1
			Next
		Else
			For Each obj In cSlingBump
				If TypeName(obj) = "Bumper" Then obj.Force = 0 Else obj.SlingshotStrength = 0
			Next
		End If
	End Sub
End Class

'--------------------
'    DropTarget
'--------------------
Class cvpmDropTarget
	Private cDropObj, cDropSw(), cDropSnd, cRaiseSnd, cSwAnyUp, cSwAllDn, cAllDn, cLink

	Private Sub Class_Initialize
		cDropSnd = 0 : cRaiseSnd = 0 : cSwAnyUp = 0 : cSwAllDn = 0 : cAllDn = False : cLink = Empty
	End sub

	Private Sub CheckAllDn(ByVal status)
		Dim obj
		If Not IsEmpty(cLink) Then
			If status Then
				For Each obj In cLink : status = status And obj.AllDown : Next
			End If
			For Each obj In cLink: obj.SetAllDn status : Next
		End If
		SetAllDn status
	End Sub

	Public Sub SetAllDn(status)
		If cSwAllDn Then Controller.Switch(cSwAllDn) = status
		If cSwAnyUp Then Controller.Switch(cSwAnyUp) = Not status
	End Sub

	Public Sub InitDrop(aWalls, aSw)
		Dim obj, obj2, ii
		' Fill in switch number
		On Error Resume Next : ReDim cDropSw(0)
		If IsArray(aSw) Then
			ReDim cDropSw(UBound(aSw))
			For ii = 0 To UBound(aSw) : cDropSw(ii) = aSw(ii) : Next
		ElseIf aSw = 0 Or Err Then
			On Error Goto 0
			If vpmIsArray(aWalls) Then
				ii = 0 : If IsArray(aWalls) Then ReDim cDropSw(UBound(aWalls)) Else ReDim cDropSw(aWalls.Count-1)
				For Each obj In aWalls
					If vpmIsArray(obj) Then
						For Each obj2 In obj
							If obj2.HasHitEvent Then cDropSw(ii) = obj2.TimerInterval : Exit For
						Next
					Else
						cDropSw(ii) = obj.TimerInterval
					End If
					ii = ii + 1
				Next
			Else
				cDropSw(0) = aWalls.TimerInterval
			End If
		Else
			cDropSw(0) = aSw
		End If
		' Copy walls
		vpmSetArray cDropObj, aWalls
	End Sub

	Public Sub CreateEvents(aName)
		Dim ii, obj1, obj2
		If Not vpmCheckEvent(aName, Me) Then Exit Sub
		ii = 1
		For Each obj1 In cDropObj
			If vpmIsArray(obj1) Then
				For Each obj2 In obj1
					If obj2.HasHitEvent Then vpmBuildEvent obj2, "Hit", aName & ".Hit " & ii
				Next
			Else
				vpmBuildEvent obj1, "Hit", aName & ".Hit " & ii
			End If
			ii = ii + 1
		Next
	End Sub

	Public Property Let AnyUpSw(aSwAnyUp)   : cSwAnyUp = aSwAnyUp : Controller.Switch(cSwAnyUp) = True : End Property
	Public Property Let AllDownSw(aSwAllDn) : cSwAllDn = aSwAllDn : End Property
	Public Property Get AllDown : AllDown = cAllDn : End Property
	Public Sub InitSnd(aDrop, aRaise) : cDropSnd = aDrop : cRaiseSnd = aRaise : End Sub
	Public Property Let LinkedTo(aLink)
		If IsArray(aLink) Then cLink = aLink Else cLink = Array(aLink)
	End Property

	Public Sub Hit(aNo)
		Dim ii
		vpmSolWall cDropObj(aNo-1), cDropSnd, True
		Controller.Switch(cDropSw(aNo-1)) = True
		For Each ii In cDropSw
			If Not Controller.Switch(ii) Then Exit Sub
		Next
		cAllDn = True : CheckAllDn True
	End Sub

	Public Sub SolHit(aNo, aEnabled) : If aEnabled Then Hit aNo : End If : End Sub

	Public Sub SolUnhit(aNo, aEnabled)
		Dim ii : If Not aEnabled Then Exit Sub
		PlaySound cRaiseSnd : vpmSolWall cDropObj(aNo-1), False, False
		Controller.Switch(cDropSw(aNo-1)) = False
		cAllDn = False : CheckAllDn False
	End Sub

	Public Sub SolDropDown(aEnabled)
		Dim ii : If Not aEnabled Then Exit Sub
		PlaySound cDropSnd
		For Each ii In cDropObj : vpmSolWall ii, False, True : Next
		For Each ii In cDropSw  : Controller.Switch(ii) = True : Next
		cAllDn = True : CheckAllDn True
	End Sub

	Public Sub SolDropUp(aEnabled)
		Dim ii : If Not aEnabled Then Exit Sub
		PlaySound cRaiseSnd
		For Each ii In cDropObj : vpmSolWall ii, False, False : Next
		For Each ii In cDropSw  : Controller.Switch(ii) = False : Next
		cAllDn = False : CheckAllDn False
	End Sub

	Public Sub DropSol_On : SolDropUp True : End Sub
End Class

'--------------------
'       Magnet
'--------------------
Class cvpmMagnet
	Private cEnabled, cBalls, cTrigger
	Public X, Y, Strength, Size, GrabCenter, Solenoid

	Private Sub Class_Initialize
		Size = 1 : Strength = 0 : Solenoid = 0 : cEnabled = False
		Set cBalls = New cvpmDictionary
	End Sub

	Private Property Let NeedUpdate(aEnabled) : vpmTimer.EnableUpdate Me, True, aEnabled : End Property

	Public Sub InitMagnet(aTrigger, aStrength)
		Dim tmp
		If vpmIsArray(aTrigger) Then Set tmp = aTrigger(0) Else Set tmp = aTrigger
		X = tmp.X : Y = tmp.Y : Size = tmp.Radius : vpmTimer.InitTimer tmp, True
		If IsArray(aTrigger) Then cTrigger = aTrigger Else Set cTrigger = aTrigger
		Strength = aStrength : GrabCenter = aStrength > 14
	End Sub

	Public Sub CreateEvents(aName)
		If vpmCheckEvent(aName, Me) Then
			vpmBuildEvent cTrigger, "Hit", aName & ".AddBall ActiveBall"
			vpmBuildEvent cTrigger, "UnHit", aName & ".RemoveBall ActiveBall"
		End If
	End Sub

	Public Property Let MagnetOn(aEnabled) : cEnabled = aEnabled : End Property
	Public Property Get MagnetOn
		If Solenoid > 0 Then MagnetOn = Controller.Solenoid(Solenoid) Else MagnetOn = cEnabled
	End Property

	Public Sub AddBall(aBall)
		With cBalls
			If .Exists(aBall) Then .Item(aBall) = .Item(aBall) + 1 Else .Add aBall, 1 : NeedUpdate = True
		End With
	End Sub

	Public Sub RemoveBall(aBall)
		With cBalls
			If .Exists(aBall) Then .Item(aBall) = .Item(aBall) - 1 : If .Item(aBall) <= 0 Then .Remove aBall
			NeedUpdate = (.Count > 0)
		End With
	End Sub

	Public Property Get Balls : Balls = cBalls.Keys : End Property

	Public Sub Update
		Dim obj
		If MagnetOn Then
			On Error Resume Next
			For Each obj In cBalls.Keys
				If obj.X < 0 Or Err Then cBalls.Remove obj Else AttractBall obj
			Next
			On Error Goto 0
		End If
	End Sub

	Public Sub AttractBall(aBall)
		Dim dX, dY, dist, force, ratio
		dX = aBall.X - X : dY = aBall.Y - Y : dist = Sqr(dX*dX + dY*dY)
		If dist > Size Or dist < 1 Then Exit Sub 'Just to be safe
		If GrabCenter And dist < 20 Then
			aBall.VelX = 0 : aBall.VelY = 0 : aBall.X = X : aBall.Y = Y
		Else
			ratio = dist / (1.5*Size)
			force = Strength * exp(-0.2/ratio)/(ratio*ratio*56) * 1.5
			aBall.VelX = (aBall.VelX - dX * force / dist) * 0.985
			aBall.VelY = (aBall.VelY - dY * force / dist) * 0.985
		End if
	End Sub
	' obsolete
	Public Property Let Range(aSize) : Size = aSize : End Property
	Public Property Get Range        : Range = Size : End Property
End Class

'--------------------
'     Turntable
'--------------------
Class cvpmTurntable
	Private cX, cY, cSize, cMotorOn, cDir, cBalls, cTrigger
	Public MaxSpeed, SpinUp, SpinDown, Speed

	Private Sub Class_Initialize
		cMotorOn = False : Speed = 0 : cDir = 1 : SpinUp = 10 : SpinDown = 4
		Set cBalls = New cvpmDictionary
	End Sub

	Private Property Let NeedUpdate(aEnabled) : vpmTimer.EnableUpdate Me, True, aEnabled : End Property

	Public Sub InitTurntable(aTrigger, aMaxSpeed)
		cX = aTrigger.X : cY = aTrigger.Y : cSize = aTrigger.Radius : vpmTimer.InitTimer aTrigger, True
		MaxSpeed = aMaxSpeed : Set cTrigger = aTrigger
	End Sub

	Public Sub CreateEvents(aName)
		If vpmCheckEvent(aName, Me) Then
			vpmBuildEvent cTrigger, "Hit", aName & ".AddBall ActiveBall"
			vpmBuildEvent cTrigger, "UnHit", aName & ".RemoveBall ActiveBall"
		End If
	End Sub

	Public Sub SolMotorState(aCW, aEnabled)
		cMotorOn = aEnabled : If aEnabled Then If aCW Then cDir = 1 Else cDir = -1
		NeedUpdate = True
	End Sub

	Public Property Let MotorOn(aEnabled)
		cMotorOn = aEnabled : NeedUpdate = cBalls.Count Or SpinUp Or SpinDown
	End Property
	Public Property Get MotorOn : MotorOn = cMotorOn : End Property

	Public Sub AddBall(aBall)
		On Error Resume Next : cBalls.Add aBall,0 : NeedUpdate = True
	End Sub
	Public Sub RemoveBall(aBall)
		On Error Resume Next
		cBalls.Remove aBall : NeedUpdate = cBalls.Count Or SpinUp Or SpinDown
	End Sub
	Public Property Get Balls : Balls = cBalls.Keys : End Property
	Public Property Let SpinCW(aCW)
		NeedUpdate = True : If aCW Then cDir = 1 Else cDir = -1
	End Property

	Public Property Get SpinCW : SpinCW = (cDir = 1) : End Property

	Public Sub Update
		If cMotorOn Then
			Speed = Speed + cDir*SpinUp/100
			If cDir * Speed >= MaxSpeed Then Speed = cDir * MaxSpeed : NeedUpdate = cBalls.Count
		Else
			Speed = Speed - cDir*SpinDown/100
			If cDir * Speed <= 0 Then Speed = 0 : NeedUpdate = cBalls.Count
		End If
		If Speed Then
			Dim obj
			On Error Resume Next
			For Each obj In cBalls.Keys
				If obj.X < 0 Or Err Then cBalls.Remove obj Else AffectBall obj
			Next
			On Error Goto 0
		End If
	End Sub

	Public Sub AffectBall(aBall)
		Dim dX, dY, dist
		dX = aBall.X - cX : dY = aBall.Y - cY : dist = Sqr(dX*dX + dY*dY)
		If dist > cSize Or dist < 1 Or Speed = 0 Then Exit Sub
		aBall.VelX = aBall.VelX - (dY * Speed / 8000)
		aBall.VelY = aBall.VelY + (dX * Speed / 8000)
	End Sub
End Class

'--------------------
'     Mech
'--------------------
Const vpmMechLinear    = &H00
Const vpmMechNonLinear = &H01
Const vpmMechCircle    = &H00
Const vpmMechStopEnd   = &H02
Const vpmMechReverse   = &H04
Const vpmMechOneSol    = &H00
Const vpmMechOneDirSol = &H10
Const vpmMechTwoDirSol = &H20
Const vpmMechStepSol   = &H40
Const vpmMechSlow      = &H00
Const vpmMechFast      = &H80
Const vpmMechStepSw    = &H00
Const vpmMechLengthSw  = &H100

Class cvpmMech
	Public Sol1, Sol2, MType, Length, Steps, Acc, Ret
	Private cMechNo, cNextSw, cSw(), cLastPos, cLastSpeed, cCallback

	Private Sub Class_Initialize
		ReDim cSw(10)
		gNextMechNo = gNextMechNo + 1 : cMechNo = gNextMechNo : cNextSw = 0 : cLastPos = 0 : cLastSpeed = 0
		MType = 0 : Length = 0 : Steps = 0 : Acc = 0 : Ret = 0 : vpmTimer.addResetObj Me
	End Sub

	Public Sub AddSw(aSwNo, aStart, aEnd)
		cSw(cNextSw) = Array(aSwNo, aStart, aEnd, 0)
		cNextSw = cNextSw + 1
	End Sub

	Public Sub AddPulseSwNew(aSwNo, aInterval, aStart, aEnd)
		If Controller.Version >= "01200000" Then
			cSw(cNextSw) = Array(aSwNo, aStart, aEnd, aInterval)
		Else
			cSw(cNextSw) = Array(aSwNo, -aInterval, aEnd - aStart + 1, 0)
		End If
		cNextSw = cNextSw + 1
	End Sub

	Public Sub Start
		Dim sw, ii
		With Controller
			.Mech(1) = Sol1 : .Mech(2) = Sol2 : .Mech(3) = Length
			.Mech(4) = Steps : .Mech(5) = MType : .Mech(6) = Acc : .Mech(7) = Ret
			ii = 10
			For Each sw In cSw
				If IsArray(sw) Then
					.Mech(ii) = sw(0) : .Mech(ii+1) = sw(1)
					.Mech(ii+2) = sw(2) : .Mech(ii+3) = sw(3)
					ii = ii + 10
				End If
			Next
			.Mech(0) = cMechNo
		End With
		If IsObject(cCallback) Then cCallBack 0, 0, 0 : cLastPos = 0 : vpmTimer.EnableUpdate Me, False, True
	End Sub

	Public Property Get Position : Position = Controller.GetMech(cMechNo) : End Property
	Public Property Get Speed    : Speed = Controller.GetMech(-cMechNo)   : End Property
	Public Property Let Callback(aCallBack) : Set cCallback = aCallBack : End Property

	Public Sub Update
		Dim currPos, speed
		currPos = Controller.GetMech(cMechNo)
		speed = Controller.GetMech(-cMechNo)
		If currPos < 0 Or (cLastPos = currPos And cLastSpeed = speed) Then Exit Sub
		cCallBack currPos, speed, cLastPos : cLastPos = currPos : cLastSpeed = speed
	End Sub

	Public Sub Reset : Start : End Sub
	' Obsolete
	Public Sub AddPulseSw(aSwNo, aInterval, aLength) : AddSw aSwNo, -aInterval, aLength : End Sub
End Class

'--------------------
'   Captive Ball
'--------------------
Class cvpmCaptiveBall
	Private cBallKicked, cBallDir, cBallCos, cBallSin, cTrigHit
	Private cTrig, cWall, cKickers, cVelX, cVelY, cKickNo
	Public ForceTrans, MinForce, RestSwitch, NailedBalls

	Private Sub Class_Initialize
		cBallKicked = False : ForceTrans = 0.5 : cTrigHit = False : MinForce = 3 : NailedBalls = 0
		vpmTimer.addResetObj Me
	End Sub

	Public Sub InitCaptive(aTrig, aWall, aKickers, aBallDir)
		Set cTrig = aTrig : Set cWall = aWall
		cKickNo = vpmSetArray(cKickers, aKickers)
		cBallDir = aBallDir : cBallCos = Cos(aBallDir * 3.1415927/180) : cBallSin = Sin(aBallDir * 3.1415927/180)
	End Sub

	Public Sub Start
		vpmCreateBall cKickers(cKickNo + (cKickNo <> NailedBalls))
		If RestSwitch Then Controller.Switch(RestSwitch) = True
	End Sub

	Public Sub TrigHit(aBall)
		cTrigHit = IsObject(aBall) : If cTrigHit Then cVelX = aBall.VelX : cVelY = aBall.VelY
	End Sub

	Public Sub Reset : If RestSwitch Then Controller.Switch(RestSwitch) = True : End If : End Sub

	Public Sub BallHit(aBall)
		Dim dX, dY, force
		If cBallKicked Then Exit Sub ' Ball is not here
		If cTrigHit Then cTrigHit = False Else cVelX = aBall.VelX : cVelY = aBall.VelY
		dX = aBall.X - cKickers(0).X : dY = aBall.Y - cKickers(0).Y
		force = -ForceTrans * (dY * cVelY + dX * cVelX) * (dY * cBallCos + dX * cBallSin) / (dX*dX + dY*dY)
		If force < 1 Then Exit Sub
		If force < MinForce Then force = MinForce
		If cKickNo <> NailedBalls Then
			vpmCreateBall cKickers(cKickNo)
			cKickers(cKickNo-1).DestroyBall
		End If
		cKickers(cKickNo).Kick cBallDir, force : cBallKicked = True
		If RestSwitch Then Controller.Switch(RestSwitch) = False
	End Sub

	Public Sub BallReturn(aKicker)
		If cKickNo <> NailedBalls Then vpmCreateBall cKickers(cKickNo-1) : aKicker.DestroyBall
		cBallKicked = False
		If RestSwitch Then Controller.Switch(RestSwitch) = True
	End Sub

	Public Sub CreateEvents(aName)
		If vpmCheckEvent(aName, Me) Then
			If Not cTrig Is Nothing Then
				vpmBuildEvent cTrig, "Hit", aName & ".TrigHit ActiveBall"
				vpmBuildEvent cTrig, "UnHit", aName & ".TrigHit 0"
			End If
			vpmBuildEvent cWall, "Hit", aName & ".BallHit ActiveBall"
			vpmBuildEvent cKickers(cKickNo), "Hit", aName & ".BallReturn Me"
		End If
	End Sub
	' Obsolete
	Public BallImage, BallColour
End Class

'--------------------
'   Visible Locks
'--------------------
Class cvpmVLock
	Private cTrig, cKick, cSw(), cSize, cBalls, cGateOpen, cRealForce, cBallSnd, cNoBallSnd
	Public ExitDir, ExitForce, KickForceVar

	Private Sub Class_Initialize
		cBalls = 0 : ExitDir = 0 : ExitForce = 0 : KickForceVar = 0 : cGateOpen = False
		vpmTimer.addResetObj Me
	End Sub

	Public Sub InitVLock(aTrig, aKick, aSw)
		Dim ii
		cSize = vpmSetArray(cTrig, aTrig)
		If vpmSetArray(cKick, aKick) <> cSize Then MsgBox "cvpmVLock: Unmatched kick+trig" : Exit Sub
		On Error Resume Next
		ReDim cSw(cSize)
		If IsArray(aSw) Then
			For ii = 0 To UBound(aSw) : cSw(ii) = aSw(ii) : Next
		ElseIf aSw = 0 Or Err Then
			For ii = 0 To cSize: cSw(ii) = cTrig(ii).TimerInterval : Next
		Else
			cSw(0) = aSw
		End If
	End Sub

	Public Sub InitSnd(aBall, aNoBall) : cBallSnd = aBall : cNoBallSnd = aNoBall : End Sub
	Public Sub CreateEvents(aName)
		Dim ii
		If Not vpmCheckEvent(aName, Me) Then Exit Sub
		For ii = 0 To cSize
			vpmBuildEvent cTrig(ii), "Hit", aName & ".TrigHit ActiveBall," & ii+1
			vpmBuildEvent cTrig(ii), "Unhit", aName & ".TrigUnhit ActiveBall," & ii+1
			vpmBuildEvent cKick(ii), "Hit", aName & ".KickHit " & ii+1
		Next
	End Sub

	Public Sub SolExit(aEnabled)
		Dim ii
		cGateOpen = aEnabled
		If Not aEnabled Then Exit Sub
		If cBalls > 0 Then PlaySound cBallSnd : Else PlaySound cNoBallSnd : Exit Sub
		For ii = 0 To cBalls-1
			cKick(ii).Enabled = False : If cSw(ii) Then Controller.Switch(cSw(ii)) = False
		Next
		If ExitForce > 0 Then ' Up
			cRealForce = ExitForce + (Rnd - 0.5)*KickForceVar : cKick(cBalls-1).Kick ExitDir, cRealForce
		Else ' Down
			cKick(0).Kick 0, 0
		End If
	End Sub

	Public Sub Reset
		Dim ii : If cBalls = 0 Then Exit Sub
		For ii = 0 To cBalls-1
			If cSw(ii) Then Controller.Switch(cSw(ii)) = True
		Next
	End Sub

	Public Property Get Balls : Balls = cBalls : End Property

	Public Property Let Balls(aBalls)
		Dim ii : cBalls = aBalls
		For ii = 0 To cSize
			If ii >= aBalls Then
				cKick(ii).DestroyBall : If cSw(ii) Then Controller.Switch(cSw(ii)) = False
			Else
				vpmCreateBall cKick(ii) : If cSw(ii) Then Controller.Switch(cSw(ii)) = True
			End If
		Next
	End Property

	Public Sub TrigHit(aBall, aNo)
		aNo = aNo - 1 : If cSw(aNo) Then Controller.Switch(cSw(aNo)) = True
		If aBall.VelY < -1 Then Exit Sub ' Allow small upwards speed
		If aNo = cSize Then cBalls = cBalls + 1
		If cBalls > aNo Then cKick(aNo).Enabled = Not cGateOpen
	End Sub

	Public Sub TrigUnhit(aBall, aNo)
		aNo = aNo - 1 : If cSw(aNo) Then Controller.Switch(cSw(aNo)) = False
		If aBall.VelY > -1 Then
			If aNo = 0 Then cBalls = cBalls - 1
			If aNo < cSize Then cKick(aNo+1).Kick 0, 0
		Else
			If aNo = cSize Then cBalls = cBalls - 1
			If aNo > 0 Then cKick(aNo-1).Kick ExitDir, cRealForce
		End If
	End Sub

	Public Sub KickHit(aNo) : cKick(aNo-1).Enabled = False : End Sub
End Class

'--------------------
'   View Dips
'--------------------
Class cvpmDips
	Private cLWF, cChkCount, cOptCount, cItems(), cVPM

	Private Sub Class_Initialize
		ReDim cItems(100)
	End Sub

	Private Sub addChkBox(aType, aLeft, aTop, aWidth, aNames)
		Dim ii, obj
		If Not isObject(cLWF) Then Exit Sub
		For ii = 0 To UBound(aNames) Step 2
			If cVPM Then
				Set obj = cLWF.AddCtrl("chkBox", 10+aLeft, 5+aTop+ii*7, aWidth, 14, aNames(ii))
			Else
				cLWF.AddCtrl "chkBox", CInt(aLeft), aTop+ii*7, CInt(aWidth), 14, CStr(aNames(ii))
				Set obj = cLWF.frmDialog.ChkBox(cChkCount + 1)
			End If
			cChkCount = cChkCount + 1 : cItems(cChkCount+cOptCount) = Array(aType, obj, cChkCount, aNames(ii+1), aNames(ii+1))
		Next
	End Sub

	Private Sub addOptBox(aType, aLeft, aTop, aWidth, aHeading, aMask, aNames)
		Dim ii, obj
		If Not isObject(cLWF) Then Exit Sub
		If cVPM Then
			cLWF.AddCtrl "Frame", 10+aLeft, 5+aTop, 10+aWidth, 7*UBound(aNames)+25, aHeading
		Else
			cLWF.AddCtrl "Frame", CInt(aLeft), CInt(aTop), aWidth+10, 7*UBound(aNames)+25, CStr(aHeading)
		End If
		If aMask Then
			For ii = 0 To UBound(aNames) Step 2
				If cVPM Then
					Set obj = cLWF.AddCtrl("OptBtn", 10+aLeft+5, 5+aTop+ii*7+14, aWidth, 14, aNames(ii))
				Else
					cLWF.AddCtrl "OptBtn", 5, ii*7+14, CInt(aWidth), 14, CStr(aNames(ii))
					Set obj = cLWF.frmDialog.Option(cOptCount + 1)
				End If
				cOptCount = cOptCount + 1 : cItems(cChkCount+cOptCount) = Array(aType+2,obj,cOptCount,aNames(ii+1),aMask)
			Next
		ElseIf cVPM Then
			addChkBox aType, 5+aLeft, 15+aTop, aWidth, aNames
		Else
			addChkBox aType, 5, 15, aWidth, aNames
		End If
	End Sub

	Public Sub addForm(ByVal aWidth, aHeight, aName)
		cVPM = Controller.Version >= "01120000"
		If aWidth < 80 Then aWidth = 80
		On Error Resume Next
		If cVPM Then
			Set cLWF = CreateObject("VPinMAME.WSHDlg") : If Err Then Exit Sub
			With cLWF
				.x = -1 : .y = -1 ' : .w = aWidth : .h = aHeight+60
				.Title = aName : .AddCtrl "OKBtn", -1, -1, 70, 25, "&Ok"
			End With
		Else
			If isObject(cLWF) Then
				cLWF.DeleteForm
			Else
				Set cLWF = CreateObject("wshLtWtForm.ucLWF") : If Err Then Exit Sub
			End If
			cLWF.CreateForm 320-aWidth\2,0,CInt(aWidth),aHeight+60,cStr(aName)
			cLWF.AddCtrl "CmdBtn", (aWidth-70)\2, CInt(aHeight), 70, 30, "&Ok"
		End If
		cChkCount = 0 : cOptCount = 0
	End Sub

	Public Sub addChk(aLeft, aTop, aWidth, aNames)
		addChkBox 0, aLeft, aTop, aWidth, aNames
	End Sub
	Public Sub addChkExtra(aLeft, aTop, aWidth, aNames)
		addChkBox 1, aLeft, aTop, aWidth, aNames
	End Sub
	Public Sub addFrame(aLeft, aTop, aWidth, aHeading, aMask, aNames)
		addOptBox 0, aLeft, aTop, aWidth, aHeading, aMask, aNames
	End Sub
	Public Sub addFrameExtra(aLeft, aTop, aWidth, aHeading, aMask, aNames)
		addOptBox 1, aLeft, aTop, aWidth, aHeading, aMask, aNames
	End Sub

	Public Sub addLabel(aLeft, aTop, aWidth, aHeight, aCaption)
		If Not isObject(cLWF) Then Exit Sub
		If cVPM Then
			cLWF.AddCtrl "Label", 10+aLeft, 5+aTop, aWidth, aHeight, aCaption
		Else
			cLWF.AddCtrl "Label",CInt(aLeft),CInt(aTop),CInt(aWidth),CInt(aHeight),CStr(aCaption)
		End If
	End Sub

	Public Sub viewDips : viewDipsExtra 0 : End Sub
	Public Function viewDipsExtra(extra)
		Dim dips(1), ii, useDip
		If Not isObject(cLWF) Then Exit Function
		With Controller
			dips(0) = .Dip(0) + .Dip(1)*256 + .Dip(2)*65536 + (.Dip(3) And &H7f)*&H1000000
			If .Dip(3) And &H80 Then dips(0) = dips(0) Or &H80000000 'workaround for overflow error
		End With
		useDip = False : dips(1) = extra
		For ii = 1 To cChkCount + cOptCount
			cItems(ii)(1).Value = -((dips(cItems(ii)(0) And &H01) And cItems(ii)(4)) = cItems(ii)(3))
			If (cItems(ii)(0) And &H01) = 0 Then useDip = True
		Next
		If cVPM Then cLWF.Show GetPlayerHWnd : Else cLWF.ShowForm
		dips(0) = 0 : dips(1) = 0
		For ii = 1 To cChkCount + cOptCount
			If cItems(ii)(1).Value Then dips(cItems(ii)(0) And &H01) = dips(cItems(ii)(0) And &H01) Or cItems(ii)(3)
		Next
		If useDip Then
			With Controller
				.Dip(0) =  (dips(0) And 255)
				.Dip(1) = ((dips(0) And 65280)\256) And 255
				.Dip(2) = ((dips(0) And &H00ff0000)\65536) And 255
				.Dip(3) = ((dips(0) And &Hff000000)\&H01000000) And 255
			End With
		End If
		viewDipsExtra = dips(1)
	End Function
End Class

Set vpmTimer = New cvpmTimer
Set vpmNudge = New cvpmNudge

'---------------------------
' Check VP version running
'---------------------------
Private Function vpmCheckVPVer
	On Error Resume Next
	' a bug in VBS?: Err object is not cleared on Exit Function
	If VPBuildVersion < 0 Or Err Then vpmCheckVPVer = 50 : Err.Clear : Exit Function
	If VPBuildVersion > 2806 Then
		vpmCheckVPVer = 63
	ElseIf VPBuildVersion > 2721 Then
		vpmCheckVPVer = 61
	Else
		vpmCheckVPVer = 60
	End If
End Function
Private vpmVPVer : vpmVPVer = vpmCheckVPVer()
'--------------------
' Initialise timers
'--------------------
Sub PulseTimer_Init  : vpmTimer.InitTimer Me, False : End Sub
Sub PinMAMETimer_Init : Me.Interval = PinMAMEInterval : Me.Enabled = True : End Sub

'---------------------------------------------
' Init function called from Table_Init event
'---------------------------------------------
Public Sub vpmInit(aTable)
	Set vpmTable = aTable
	If vpmVPVer >= 60 Then
		On Error Resume Next
		If Not IsObject(GetRef(aTable.Name & "_Paused")) Or Err Then
			vpmBuildEvent aTable, "Paused",   "Controller.Pause = True"
			vpmBuildEvent aTable, "UnPaused", "Controller.Pause = False"
		End If
	End If
End Sub

' Exit function called in Table_Exit event
Public Sub vpmExit : End Sub
'------------------------------------------------------
' All classes call this function to create a ball
' Assign vpmCreateBall if you want a custom function
'------------------------------------------------------
Private Function vpmDefCreateBall(aKicker)
	If Not IsEmpty(vpmBallImage) Then aKicker.CreateBall.Image = vpmBallImage Else aKicker.CreateBall : End If
	Set vpmDefCreateBall = aKicker
End Function
Set vpmCreateBall = GetRef("vpmDefCreateBall")
Private vpmTrough ' Default Trough. Used to clear up missing balls
Private vpmTable  ' Table object

'-------------------
' Main Loop
'------------------
Private Const CHGNO = 0
Private Const CHGSTATE = 1
Private vpmTrueFalse : vpmTrueFalse = Array(" True", " False"," True")

Sub vpmDoSolCallback(aNo, aEnabled)
	If SolCallback(aNo) <> "" Then Execute SolCallback(aNo) & vpmTrueFalse(aEnabled+1)
End Sub

Sub vpmDoLampUpdate(aNo, aEnabled)
	On Error Resume Next : Lights(aNo).State = Abs(aEnabled)
End Sub

Sub PinMAMETimer_Timer
	Dim ChgLamp,ChgSol,ChgGI, ii, tmp, idx
	Me.Enabled = False
	On Error Resume Next
		If UseLamps Then ChgLamp = Controller.ChangedLamps Else LampCallback
		If UseSolenoids Then ChgSol  = Controller.ChangedSolenoids
		If isObject(GICallback) Then ChgGI = Controller.ChangedGIStrings
		MotorCallback
	On Error Goto 0
	If Not IsEmpty(ChgLamp) Then
		On Error Resume Next
			For ii = 0 To UBound(ChgLamp)
				idx = chgLamp(ii, 0)
				If IsArray(Lights(idx)) Then
					For Each tmp In Lights(idx) : tmp.State = ChgLamp(ii, 1) : Next
				Else
					Lights(idx).State = ChgLamp(ii, 1)
				End If
			Next
			For Each tmp In vpmMultiLights
				For ii = 1 To UBound(tmp) : tmp(ii).State = tmp(0).State : Next
			Next
			LampCallback
		On Error Goto 0
	End If
	If Not IsEmpty(ChgSol) Then
		For ii = 0 To UBound(ChgSol)
			tmp = SolCallback(ChgSol(ii, 0))
			If tmp <> "" Then Execute tmp & vpmTrueFalse(ChgSol(ii, 1)+1)
		Next
	End If
	If Not IsEmpty(ChgGI) Then
		For ii = 0 To UBound(ChgGI)
			GICallback ChgGI(ii, 0), CBool(ChgGI(ii, 1))
		Next
	End If
	Me.Enabled = True
End Sub

'
' Private helper functions
'
Private Sub vpmPlaySound(aEnabled, aSound)
	If VarType(aSound) = vbString Then
		If aEnabled Then StopSound aSound : PlaySound aSound
	ElseIf aSound Then
		If aEnabled Then PlaySound SSolenoidOn Else PlaySound SSolenoidOff
	End If
End Sub

Private Sub vpmToggleObj(aObj, aEnabled)
	Select Case TypeName(aObj)
		Case "Wall"                        aObj.IsDropped = aEnabled
		Case "Bumper", "Light"             aObj.State     = Abs(aEnabled)
		Case "Kicker", "Trigger", "Timer"  aObj.Enabled   = aEnabled
		Case "Gate"                        aObj.Open      = aEnabled
		Case "Integer"                     Controller.Switch(aObj) = aEnabled
		Case Else MsgBox "vpmToggleObj: Unhadled Object " & TypeName(aObj)
	End Select
End Sub

Private Function vpmCheckEvent(aName, obj)
	vpmCheckEvent = True
	On Error Resume Next
	If Not Eval(aName) Is obj Or Err Then MsgBox "CreateEvents: Wrong name " & aName : vpmCheckEvent = False
End Function

Private Sub vpmBuildEvent(aObj, aEvent, aTask)
	Dim obj, str
	str = "_" & aEvent & " : " & aTask & " : End Sub"
	If vpmIsArray(aObj) Then
		For Each obj In aObj : ExecuteGlobal "Sub " & obj.Name & str : Next
	Else
		ExecuteGlobal "Sub " & aObj.Name & str
	End If
End Sub

Private Function vpmIsCollection(aObj)
	vpmIsCollection =  TypeName(aObj) = "Collection" Or TypeName(aObj) = "ICollection"
End Function
Private Function vpmIsArray(aObj)
	vpmIsArray = IsArray(aObj) Or vpmIsCollection(aObj)
End Function

Private Function vpmSetArray(aTo, aFrom)
	If IsArray(aFrom) Then
		aTo = aFrom : vpmSetArray = UBound(aFrom)
	ElseIf vpmIsCollection(aFrom) Then
		Set aTo = aFrom : vpmSetArray = aFrom.Count - 1
	Else
		aTo = Array(aFrom) : vpmSetArray = 0
	End If
End Function

Sub vpmCreateEvents(aHitObjs)
	Dim obj
	For Each obj In aHitObjs
		Select Case TypeName(obj)
			Case "Trigger"
				vpmBuildEvent obj, "Hit", "Controller.Switch(" & Obj.TimerInterval & ") = True"
				vpmBuildEvent obj, "UnHit", "Controller.Switch(" & Obj.TimerInterval & ") = False"
			Case "Wall"
				If obj.HasHitEvent Then
					vpmBuildEvent obj, "Hit", "vpmTimer.PulseSw " & Obj.TimerInterval
				Else
					vpmBuildEvent obj, "SlingShot", "vpmTimer.PulseSw " & Obj.TimerInterval
				End If
			Case "Bumper", "Gate"
				vpmBuildEvent obj, "Hit","vpmTimer.PulseSw " & Obj.TimerInterval
			Case "Spinner"
				vpmBuildEvent obj, "Spin","vpmTimer.PulseSw " & Obj.TimerInterval
		End Select
	Next
End Sub

Sub vpmMapLights(aLights)
	Dim obj, str, ii, idx
	For Each obj In aLights
		idx = obj.TimerInterval
		If IsArray(Lights(idx)) Then
			str = "Lights(" & idx & ") = Array("
			For Each ii In Lights(idx) : str = str & ii.Name & "," : Next
			ExecuteGlobal str & obj.Name & ")"
		ElseIf IsObject(Lights(idx)) Then
            Lights(idx) = Array(Lights(idx),obj)
		Else
			Set Lights(idx) = obj
		End If
	Next
End Sub

Function vpmMoveBall(aBall, aFromKick, aToKick)
	With aToKick.CreateBall
		If TypeName(aBall) = "IBall" Then
			.Color = aBall.Color   : .Image = aBall.Image
			If vpmVPVer >= 60 Then
				.FrontDecal = aBall.FrontDecal : .BackDecal = aBall.BackDecal
'				.UserValue = aBall.UserValue
			End If
		End If
	End With
	aFromKick.DestroyBall : Set vpmMoveBall = aToKick
End Function

Sub vpmAddBall
	If IsObject(vpmTrough) Then
		If MsgBox("This will insert a new ball on the table" & vbNewLine &_
		          "Are you sure?",vbYesNo + vbQuestion) = vbYes Then vpmTrough.AddBall 0
	End If
End Sub

'----------------------------
' Generic solenoid handlers
'----------------------------
' ----- Flippers ------
Sub vpmSolFlipper(aFlip1, aFlip2, aEnabled)
	Dim oldSpeed
	If aEnabled Then
		PlaySound SFlipperOn : aFlip1.RotateToEnd : If Not aFlip2 Is Nothing Then aFlip2.RotateToEnd
	Else
		PlaySound SFlipperOff
		oldSpeed = aFlip1.Speed : aFlip1.Speed = 0.03
		aFlip1.RotateToStart : aFlip1.Speed = oldSpeed
		If Not aFlip2 Is Nothing Then
			oldSpeed = aFlip2.Speed : aFlip2.Speed = 0.03
			aFlip2.RotateToStart : aFlip2.Speed = oldSpeed
		End If
	End If
End Sub

' ------ Diverters ------
Sub vpmSolDiverter(aDiv, aSound, aEnabled)
	If aEnabled Then aDiv.RotateToEnd : Else aDiv.RotateToStart
	vpmPlaySound aEnabled, aSound
End sub

' ------ Walls ------
Sub vpmSolWall(aWall, aSound, aEnabled)
	Dim obj
	If vpmIsArray(aWall) Then
		For Each obj In aWall : obj.IsDropped = aEnabled : Next
	Else
		aWall.IsDropped = aEnabled
	End If
	vpmPlaySound aEnabled, aSound
End Sub

Sub vpmSolToggleWall(aWall1, aWall2, aSound, aEnabled)
	Dim obj
	If vpmIsArray(aWall1) Then
		For Each obj In aWall1 : obj.IsDropped = aEnabled : Next
	Else
		aWall1.IsDropped = aEnabled
	End If
	If vpmIsArray(aWall2) Then
		For Each obj In aWall2 : obj.IsDropped = Not aEnabled : Next
	Else
		aWall2.IsDropped = Not aEnabled
	End If
	vpmPlaySound aEnabled, aSound
End Sub

' ------- Autoplunger ------
Sub vpmSolAutoPlunger(aPlung, aVar, aEnabled)
	Dim oldFire
	If aEnabled Then
		oldFire = aPlung.FireSpeed : aPlung.FireSpeed = oldFire * (100-aVar*(2*Rnd-1))/100
		PlaySound SSolenoidOn : aPlung.Fire : aPlung.FireSpeed = oldFire
	Else
		aPlung.Pullback
	End If
End Sub

' --------- Gate -----------
Sub vpmSolGate(aGate, aSound, aEnabled)
	Dim obj
	If vpmIsArray(aGate) Then
		For Each obj In aGate : obj.Open = aEnabled : Next
	Else
		aGate.Open = aEnabled
	End If
	vpmPlaySound aEnabled, aSound
End Sub

' ------ Sound Only -------
Sub vpmSolSound(aSound, aEnabled)
	If aEnabled Then StopSound aSound : PlaySound aSound
End Sub

' ------- Flashers --------
Sub vpmFlasher(aFlash, aEnabled)
	Dim obj
	If vpmIsArray(aFlash) Then
		For Each obj In aFlash : obj.State = Abs(aEnabled) : Next
	Else
		aFlash.State = Abs(aEnabled)
	End If
End Sub

'---- Generic object toggle ----
Sub vpmSolToggleObj(aObj1, aObj2, aSound, aEnabled)
	Dim obj
	If vpmIsArray(aObj1) Then
		If IsArray(aObj1(0)) Then
			For Each obj In aObj1(0) : vpmToggleObj obj, aEnabled     : Next
			For Each obj In aObj1(1) : vpmToggleObj obj, Not aEnabled : Next
		Else
			For Each obj In aObj1    : vpmToggleObj obj, aEnabled     : Next
		End If
	ElseIf Not aObj1 Is Nothing Then
		vpmToggleObj aObj1, aEnabled
	End If
	If vpmIsArray(aObj2) Then
		If IsArray(aObj2(0)) Then
			For Each obj In aObj2(0) : vpmToggleObj obj, Not aEnabled : Next
			For Each obj In aObj2(1) : vpmToggleObj obj, aEnabled     : Next
		Else
			For Each obj In aObj2    : vpmToggleObj obj, Not aEnabled : Next
		End If
	ElseIf Not aObj2 Is Nothing Then
		vpmToggleObj aObj2, Not aEnabled
	End If
	vpmPlaySound aEnabled, aSound
End Sub

'
' Stubs to allow older games to still work
' These will be removed one day
'
Sub SolFlipper(f1,f2,e) : vpmSolFlipper f1,f2,e : End Sub
Sub SolDiverter(d,s,e) : vpmSolDiverter d,s,e : End Sub
Sub SolSound(s,e) : vpmSolSound s,e : End Sub
Sub Flasher(f,e) : vpmFlasher f,e : End Sub
Sub SolMagnet(m,e) : vpmSolMagnet m,e : End Sub
Sub SolAutoPlunger(p,e) : vpmSolAutoPlunger p,0,e : End Sub
Function KeyDownHandler(ByVal k) : KeyDownHandler = vpmKeyDown(k) : End Function
Function KeyUpHandler(ByVal k) : KeyUpHandler = vpmKeyUp(k) : End Function
Function KeyName(ByVal k) : KeyName = vpmKeyName(k) : End Function
Sub vpmSolMagnet(m,e) : m.Enabled = e : If Not e Then m.Kick 180,1 : End If : End Sub
Dim vpmBallImage : vpmBallImage = Empty ' Default ball properties
Dim vpmBallColour

'-- Flipper solenoids (all games)
Const sLRFlipper = 46
Const sLLFlipper = 48
Const sURFlipper = 34
Const sULFlipper = 36

' Convert keycode to readable string
Private keyNames1, keyNames2
keyNames1 = Array("Escape","1","2","3","4","5","6","7","8","9","0","Minus '-'",_
"Equals '='","Backspace","Tab","Q","W","E","R","T","Y","U","I","O","P","[","]",_
"Enter","Left Ctrl","A","S","D","F","G","H","J","K","L",";","'","`","Left Shift",_
"\","Z","X","C","V","B","N","M",",",".","/","Right Shift","*","Left Menu","Space",_
"Caps Lock","F1","F2","F3","F4","F5","F6","F7","F8","F9","F10","NumLock","ScrlLock",_
"Numpad 7","Numpad 8","Numpad 9","Numpad -","Numpad 4","Numpad 5","Numpad 6",_
"Numpad +","Numpad 1","Numpad 2","Numpad 3","Numpad 0","Numpad .","?","?","?",_
"F11","F12","F13","F14","F15")
keyNames2 = Array("Pause","?","Home","Up","PageUp","?","Left","?","Right","?",_
"End","Down","PageDown","Insert","Delete")

Function vpmKeyName(ByVal aKeycode)
	If aKeyCode-1 <= UBound(keyNames1) Then
		vpmKeyName = keyNames1(aKeyCode-1)
	ElseIf aKeyCode >= 197 And aKeyCode <= 211 Then
		vpmKeyName = keyNames2(aKeyCode-197)
	ElseIf aKeyCode = 184 Then
		vpmKeyName = "R.Alt"
	Else
		vpmKeyName = "?"
	End If
End Function

Private vpmSystemHelp
Private Sub vpmShowHelp
	Dim szKeyMsg
	szKeyMsg = "The following keys are defined: "                  & vbNewLine &_
	           "(American keyboard layout)"                        & vbNewLine &_
		vbNewLine & "Visual PinMAME keys:"                         & vbNewLine &_
		vpmKeyName(keyShowOpts)   & vbTab & "Game options..."      & vbNewLine &_
		vpmKeyName(keyShowKeys)   & vbTab & "Keyboard settings..." & vbNewLine &_
		vpmKeyName(keyReset)      & vbTab & "Reset emulation"      & vbNewLine &_
		vpmKeyName(keyFrame)      & vbTab & "Toggle Display lock"  & vbNewLine &_
		vpmKeyName(keyDoubleSize) & vbTab & "Toggle Display size"  & vbNewLine
	If IsObject(vpmShowDips) Then
			szKeyMsg = szKeyMsg & vpmKeyName(keyShowDips)   & vbTab & "Show DIP switches" & vbNewLine
		End If
	If IsObject(vpmTrough) Then
		szKeyMsg = szKeyMsg & vpmKeyName(keyAddBall) & vbTab & "Replace lost ball" & vbNewLine
	End If
	szKeyMsg = szKeyMsg & vpmKeyName(keyBangBack) & vbTab & "Bang Back" & vbNewLine &_
		vbNewLine & vpmSystemHelp & vbNewLine
	If ExtraKeyHelp <> "" Then
		szKeyMsg = szKeyMsg & vbNewLine & "Game Specific keys:" &_
			vbNewLine & ExtraKeyHelp & vbNewLine
	End If
	szKeyMsg = szKeyMsg & vbNewLine & "Visual Pinball keys:"     & vbNewLine &_
		vpmKeyName(LeftFlipperKey)  & vbTab & "Left Flipper"     & vbNewLine &_
		vpmKeyName(RightFlipperKey) & vbTab & "Right Flipper"    & vbNewLine &_
		vpmKeyName(PlungerKey)      & vbTab & "Launch Ball"      & vbNewLine &_
		vpmKeyName(StartGameKey)    & vbTab & "Start Button"     & vbNewLine &_
		vpmKeyName(LeftTiltKey)     & vbTab & "Nudge from Left"  & vbNewLine &_
		vpmKeyName(RightTiltKey)    & vbTab & "Nudge from Right" & vbNewLine &_
		vpmKeyName(CenterTiltKey)   & vbTab & "Nudge forward"    & vbNewLine
	MsgBox szKeyMsg,vbOkOnly,"Keyboard Settings..."
End Sub

